#include <node_api.h>

#include <stdlib.h>
//#include <stdio.h>

void crcReflectTable(uint64_t poly64Rev, uint64_t* lookupTable) {
        uint16_t i, j;
        uint64_t v;
        for(i = 0; i < 256; ++i) {
                v = i;
                for(j = 0; j < 8; ++j) {
                        if((v & 1) == 1) {
                                v >>= 1;
                                v ^= poly64Rev;
                        }else{
                                v >>= 1;
                        }
                }
                lookupTable[i] = v;
        }
}

void crcTable(uint64_t poly64, uint64_t* lookupTable, uint8_t bits){
        uint64_t mask1 = (uint64_t)1 << (bits - 1);
        uint64_t mask2 = 1;

        uint16_t i, j;
        uint64_t v;
        for(i = 1; i < bits; ++i) {
                mask2 = (mask2 << 1) + 1;
        }
        for(i = 0; i < 256; ++i) {
                v = i;
                for(j = 0; j < bits; ++j) {
                        if((v & mask1) == 0) {
                                v <<= 1;
                        }else{
                                v <<= 1;
                                v ^= poly64;
                        }
                }
                lookupTable[i] = v & mask2;
        }
}

uint64_t crcInternalWithLookUpTable(uint64_t* lookupTable, char* data, size_t dataLength, uint64_t initial, uint64_t finalXor, bool reflection, uint8_t bits){
        uint64_t sum = initial, i;
        uint8_t index;
        if(reflection || bits <= 8) {
                for(i = 0; i < dataLength; ++i) {
                        index = (sum ^ data[i]);
                        sum = (sum >> 8) ^ lookupTable[index];
                }
        }else{
                for(i = 0; i < dataLength; ++i) {
                        index = ((sum >> (bits - 8)) ^ data[i]);
                        sum = (sum << 8) ^ lookupTable[index];
                }
        }
        sum ^= finalXor;
        return sum;
}

uint64_t crcInternal(uint64_t expression, char* data, size_t dataLength, uint64_t initial, uint64_t finalXor, bool reflection, uint8_t bits){
        uint64_t lookupTable[256];
        if(reflection) {
                crcReflectTable(expression, lookupTable);
        }else{
                crcTable(expression, lookupTable, bits);
        }
/*
        printf("{");
        for(int i = 0; i < 255; ++i) {
                printf("%ld,", lookupTable[i]);
        }
        printf("%ld", lookupTable[255]);
        printf("}\n");
 */
        return crcInternalWithLookUpTable(lookupTable, data, dataLength, initial, finalXor, reflection, bits);
}

napi_value createFalse(napi_env env){
        napi_value result;
        napi_get_boolean(env, false, &result);
        return result;
}

napi_value toBuffer(napi_env env, uint64_t value, uint8_t bits){
        size_t byte = ((size_t)(bits + 7)) / 8;
        napi_value buffer;
        char* data;
        napi_create_buffer(env, byte, (void**)(&data), &buffer);
        int64_t i = byte - 1;
        for(; i >= 0; --i) {
                data[i] = value & 0xff;
                value >>= 8;
        }
        return buffer;
}

napi_value crc(napi_env env, napi_callback_info info){
        size_t argsLength = 9;
        napi_value args[9];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 9) {
                return createFalse(env);
        }

        uint32_t expL, expH, iniL, iniH, fixL, fixH, bits;
        uint64_t expression, initial, finalXor;
        char* data;
        size_t dataLength;
        bool reflection;

        napi_get_value_uint32(env, args[0], &bits);

        napi_get_value_bool(env, args[1], &reflection);

        napi_get_value_uint32(env, args[2], &expL);
        napi_get_value_uint32(env, args[3], &expH);
        expression = ((uint64_t)expH << 32) | expL;

        napi_get_value_uint32(env, args[4], &iniL);
        napi_get_value_uint32(env, args[5], &iniH);
        initial = ((uint64_t)iniH << 32) | iniL;

        napi_get_value_uint32(env, args[6], &fixL);
        napi_get_value_uint32(env, args[7], &fixH);
        finalXor = ((uint64_t)fixH << 32) | fixL;

        napi_get_buffer_info(env, args[8], (void**)(&data), &dataLength);

        uint64_t sum = crcInternal(expression, data, dataLength, initial, finalXor, reflection, bits);
        return toBuffer(env, sum, bits);
}

napi_value crc64iso(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,121597189939003392,243194379878006784,202661983231672320,486388759756013568,535928355657089024,405323966463344640,292733975779082240,972777519512027136,878201927337246720,1071856711314178048,1103381908705771520,810647932926689280,788129934789836800,585467951558164480,689050742987685888,1945555039024054272,1923037040887201792,1756403854674493440,1859986646104014848,2143713422628356096,2049137830453575680,2206763817411543040,2238289014803136512,1621295865853378560,1670835461754454016,1576259869579673600,1463669878895411200,1170935903116328960,1292533093055332352,1378101485975371776,1337569089329037312,3891110078048108544,4012707267987111936,3846074081774403584,3805541685128069120,3512807709348986880,3562347305250062336,3719973292208029696,3607383301523767296,4287426845256712192,4192851253081931776,4098275660907151360,4129800858298744832,4413527634823086080,4391009636686233600,4476578029606273024,4580160821035794432,3242591731706757120,3220073733569904640,3341670923508908032,3445253714938429440,3152519739159347200,3057944146984566784,2927339757790822400,2958864955182415872,2341871806232657920,2391411402133733376,2585066186110664704,2472476195426402304,2756202971950743552,2877800161889746944,2675138178658074624,2634605782011740160,7782220156096217088,7903817346035220480,8025414535974223872,7984882139327889408,7692148163548807168,7741687759449882624,7611083370256138240,7498493379571875840,7025615418697973760,6931039826523193344,7124694610500124672,7156219807891718144,7439946584416059392,7417428586279206912,7214766603047534592,7318349394477056000,8574853690513424384,8552335692376571904,8385702506163863552,8489285297593384960,8196551321814302720,8101975729639522304,8259601716597489664,8291126913989083136,8827055269646172160,8876594865547247616,8782019273372467200,8669429282688204800,8953156059212546048,9074753249151549440,9160321642071588864,9119789245425254400,6485183463413514240,6606780653352517632,6440147467139809280,6399615070493474816,6683341847017816064,6732881442918891520,6890507429876858880,6777917439192596480,6305039478318694400,6210463886143913984,6115888293969133568,6147413491360727040,5854679515581644800,5832161517444792320,5917729910364831744,6021312701794353152,4683743612465315840,4661225614328463360,4782822804267466752,4886405595696988160,5170132372221329408,5075556780046548992,4944952390852804608,4976477588244398080,5512405943901487104,5561945539802562560,5755600323779493888,5643010333095231488,5350276357316149248,5471873547255152640,5269211564023480320,5228679167377145856,-2882303761517117440,-2760706571578114048,-2639109381639110656,-2679641778285445120,-2395915001761103872,-2346375405860028416,-2476979795053772800,-2589569785738035200,-3062447746611937280,-3157023338786717696,-2963368554809786368,-2931843357418192896,-3224577333197275136,-3247095331334127616,-3449757314565799936,-3346174523136278528,-4395513236313604096,-4418031234450456576,-4584664420663164928,-4481081629233643520,-4197354852709302272,-4291930444884082688,-4134304457926115328,-4102779260534521856,-3566850904877432832,-3517311308976357376,-3611886901151137792,-3724476891835400192,-4017210867614482432,-3895613677675479040,-3810045284755439616,-3850577681401774080,-1297036692682702848,-1175439502743699456,-1342072688956407808,-1382605085602742272,-1675339061381824512,-1625799465480749056,-1468173478522781696,-1580763469207044096,-2053641430080946176,-2148217022255726592,-2242792614430507008,-2211267417038913536,-1927540640514572288,-1950058638651424768,-1864490245731385344,-1760907454301863936,-792633534417207296,-815151532554059776,-693554342615056384,-589971551185534976,-882705526964617216,-977281119139397632,-1107885508333142016,-1076360310941548544,-540431955284459520,-490892359383384064,-297237575406452736,-409827566090715136,-126100789566373888,-4503599627370496,-207165582859042816,-247697979505377280,-5476377146882523136,-5354779956943519744,-5233182767004516352,-5273715163650850816,-5566449139429933056,-5516909543528857600,-5647513932722601984,-5760103923406864384,-5080060379673919488,-5174635971848699904,-4980981187871768576,-4949455990480175104,-4665729213955833856,-4688247212092686336,-4890909195324358656,-4787326403894837248,-5836665117072162816,-5859183115209015296,-6025816301421723648,-5922233509992202240,-6214967485771284480,-6309543077946064896,-6151917090988097536,-6120391893596504064,-6737385042546262016,-6687845446645186560,-6782421038819966976,-6895011029504229376,-6611284252979888128,-6489687063040884736,-6404118670120845312,-6444651066767179776,-9079256848778919936,-8957659658839916544,-9124292845052624896,-9164825241698959360,-8881098465174618112,-8831558869273542656,-8673932882315575296,-8786522872999837696,-8106479329266892800,-8201054921441673216,-8295630513616453632,-8264105316224860160,-8556839292003942400,-8579357290140794880,-8493788897220755456,-8390206105791234048,-7421932185906577408,-7444450184043429888,-7322852994104426496,-7219270202674905088,-6935543426150563840,-7030119018325344256,-7160723407519088640,-7129198210127495168,-7746191359077253120,-7696651763176177664,-7502996979199246336,-7615586969883508736,-7908320945662590976,-7786723755723587584,-7989385738955259904,-8029918135601594368};

        //uint64_t sum = crcInternal(0xD800000000000000, data, dataLength, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, true, 64);
        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, true, 64);
        return toBuffer(env, sum, 64);
}

napi_value crc64ecma(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,4823603603198064275,-8799536867313423066,-4102460140266038347,5274672035359026399,847670339082705484,-3687703096809061895,-8204920280532076694,-7897400002991498818,-3416493369634853075,1695340678165410968,6158653484774949387,-2642017800032930463,-7375406193618123790,6824194888265062471,2036903512645398228,7367177604490692079,2651944067726553980,-2027539948475389751,-6832986739269706150,3390681356330821936,7926053118503640995,-6129437104159652842,-1720589984720932219,-838878488614904751,-5284035600065860926,8194994013375312247,3695931686473304036,-4798354297179426674,-29216381152229859,4073807025290796456,8825348881154370363,-3712388864728167458,-8175704493167919795,5303888135453107960,822984195088142443,-8842369567448504575,-4055079896950779502,47380625301539367,4780770595170139316,6781362712661643872,2084283301222999283,-2594637836702269626,-7418238609469699627,1670654249350217407,6187869865390245932,-7868183379440544871,-3441179969441864438,-6176817727850508751,-1677756977229809502,3433514057002836759,7878672873577829764,-2056756046958927122,-6808300595812084611,7391863372946608072,2622728278751721819,4044590402276644751,8850035479350698268,-4773667866753680727,-58432762304459718,8147614050581592912,3738764100714335683,-796046311400810890,-5331415389180272411,-2736778905407184593,-7424777729456334916,6909860770376862729,2095335087373712026,-7838967802803335696,-3330827834883769501,1645968390176284886,6063892853452478021,5216239979862816913,762004938812542466,-3638330943300856393,-8110159793901559004,94761250603078734,4872975272980325085,-8885202883369272984,-4160891860223176709,-4884018648386263872,-87649760589671853,4168566602445998566,8874722219015798645,-789505769768794081,-5189275673404539252,8136561383943382329,3610266854770152362,3341308498700434814,7831293060043656173,-6071004342929059752,-1634925014233504053,7452841817450123681,2710377314828461874,-2122299393295057785,-6882359938883728876,1621282580641819377,6093108618008534114,-7809751662704506921,-3355513954459619004,6867028114005673518,2142715359940571325,-2689398326553892088,-7467610764051505765,-8928035101125971121,-4113512093917854244,142141253402664553,4830142882085382394,-3663017327816335472,-8080943384572581629,5245456557503443638,737318311902463013,8089180804553289502,3653099890976004493,-746673115008155080,-5236655945434467157,4139350461810230209,8899408340202190162,-4859332840462471449,-116865524608919436,-2151515972546365792,-6857673311436849101,7477528201428671366,2681160907110034709,-6118384347339187585,-1592092622801621780,3384140715920324441,7783913295349006794,-649954581304675123,-5473557810814369186,8294265019745835499,3597188614796881784,-4627022532955826158,-200020480187781503,4190670174747424052,8707887697765516199,7249714899603402099,2768808468102880224,-2198343082210770859,-6661655669767539002,3291936780352569772,8025325358597240639,-6318958366804595574,-1531666754935514599,-8014264113983917790,-3299030950906050639,1524009877625084932,6329456346323069591,-2741289767939269123,-7276661886601712786,6635271944638132443,2226424485906433608,189522501206157468,4634679410803088911,-8700793527748901446,-4201731419897564375,5445476407655580739,676338306971005648,-3570241628335462043,-8321783720446353418,4215391513593610003,8678706776937023872,-4656203148037909963,-175299521179343706,8337133204891997132,3549843186494580063,-697299635677954326,-5430689936249599879,-6276090757712140627,-1579011539537588162,3244592164593781643,8068192726900473112,-2173621305822786958,-6690837097930261279,7220533709540304724,2793530071884239303,6682616997400869628,2183556611878603887,-2784157953622239270,-7229316456688737975,1553190491096487459,6304735387851432112,-8038985453367035131,-3269850028467008106,-3541060438809304254,-8346505322617170479,5420754629656923748,705519735670536439,-8653448912526913635,-4244598786590115570,146654890503152315,4682024195942093864,3242565161283638754,7930564333232481137,-6260526837692483388,-1446000823986287017,7335380351123765565,2827240748300537774,-2293103759149444069,-6711027908919238008,-4712687845698204580,-258452628580484401,4285430719881142650,8757259798139230185,-600582823994630013,-5378796653107784176,8235833358291897765,3511522545606540086,5387043107155988493,590673871457609374,-3520868240560767701,-8227024187835708488,284282506805329106,4684052045342640705,-8786458309538786828,-4260164093874051225,-2836049918219908685,-7326034655632670944,6720936860919424149,2284857304564388358,-7955830958702664340,-3213366649347189761,1474636623804926026,6234696958930763481,-2268382464602972612,-6740208858461411153,7306199781952008986,2851961734412043657,-6217658610393042205,-1493346230016310160,3195220067441434565,7973432182840617302,8278700923620460418,3464177731752866065,-647927393305171292,-5335929258237305801,4310152537884486493,8728078392784608718,-4741869075766049157,-233731049217838872,-8739113215159641133,-4303031945092731584,241414281116563189,4731397450835853414,-3491687670852208884,-8251745175557897825,5362321814220069418,619854820462849209,1503817855483314797,6209975379031176446,-7980552776169197749,-3184185245603243560,6768281431840648882,2241989909157107745,-2878917483011538028,-7278689843389549305};

        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, false, 64);
        return toBuffer(env, sum, 64);
}

napi_value crc32ieee(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,1996959894,3993919788,2567524794,124634137,1886057615,3915621685,2657392035,249268274,2044508324,3772115230,2547177864,162941995,2125561021,3887607047,2428444049,498536548,1789927666,4089016648,2227061214,450548861,1843258603,4107580753,2211677639,325883990,1684777152,4251122042,2321926636,335633487,1661365465,4195302755,2366115317,997073096,1281953886,3579855332,2724688242,1006888145,1258607687,3524101629,2768942443,901097722,1119000684,3686517206,2898065728,853044451,1172266101,3705015759,2882616665,651767980,1373503546,3369554304,3218104598,565507253,1454621731,3485111705,3099436303,671266974,1594198024,3322730930,2970347812,795835527,1483230225,3244367275,3060149565,1994146192,31158534,2563907772,4023717930,1907459465,112637215,2680153253,3904427059,2013776290,251722036,2517215374,3775830040,2137656763,141376813,2439277719,3865271297,1802195444,476864866,2238001368,4066508878,1812370925,453092731,2181625025,4111451223,1706088902,314042704,2344532202,4240017532,1658658271,366619977,2362670323,4224994405,1303535960,984961486,2747007092,3569037538,1256170817,1037604311,2765210733,3554079995,1131014506,879679996,2909243462,3663771856,1141124467,855842277,2852801631,3708648649,1342533948,654459306,3188396048,3373015174,1466479909,544179635,3110523913,3462522015,1591671054,702138776,2966460450,3352799412,1504918807,783551873,3082640443,3233442989,3988292384,2596254646,62317068,1957810842,3939845945,2647816111,81470997,1943803523,3814918930,2489596804,225274430,2053790376,3826175755,2466906013,167816743,2097651377,4027552580,2265490386,503444072,1762050814,4150417245,2154129355,426522225,1852507879,4275313526,2312317920,282753626,1742555852,4189708143,2394877945,397917763,1622183637,3604390888,2714866558,953729732,1340076626,3518719985,2797360999,1068828381,1219638859,3624741850,2936675148,906185462,1090812512,3747672003,2825379669,829329135,1181335161,3412177804,3160834842,628085408,1382605366,3423369109,3138078467,570562233,1426400815,3317316542,2998733608,733239954,1555261956,3268935591,3050360625,752459403,1541320221,2607071920,3965973030,1969922972,40735498,2617837225,3943577151,1913087877,83908371,2512341634,3803740692,2075208622,213261112,2463272603,3855990285,2094854071,198958881,2262029012,4057260610,1759359992,534414190,2176718541,4139329115,1873836001,414664567,2282248934,4279200368,1711684554,285281116,2405801727,4167216745,1634467795,376229701,2685067896,3608007406,1308918612,956543938,2808555105,3495958263,1231636301,1047427035,2932959818,3654703836,1088359270,936918000,2847714899,3736837829,1202900863,817233897,3183342108,3401237130,1404277552,615818150,3134207493,3453421203,1423857449,601450431,3009837614,3294710456,1567103746,711928724,3020668471,3272380065,1510334235,755167117};

        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0x00000000FFFFFFFF, 0x00000000FFFFFFFF, true, 32);
        return toBuffer(env, sum, 32);
}

napi_value crc32mhash(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,79764919,159529838,222504665,319059676,398814059,445009330,507990021,638119352,583659535,797628118,726387553,890018660,835552979,1015980042,944750013,1276238704,1221641927,1167319070,1095957929,1595256236,1540665371,1452775106,1381403509,1780037320,1859660671,1671105958,1733955601,2031960084,2111593891,1889500026,1952343757,2552477408,2632100695,2443283854,2506133561,2334638140,2414271883,2191915858,2254759653,3190512472,3135915759,3081330742,3009969537,2905550212,2850959411,2762807018,2691435357,3560074640,3505614887,3719321342,3648080713,3342211916,3287746299,3467911202,3396681109,4063920168,4143685023,4223187782,4286162673,3779000052,3858754371,3904687514,3967668269,881225847,809987520,1023691545,969234094,662832811,591600412,771767749,717299826,311336399,374308984,453813921,533576470,25881363,88864420,134795389,214552010,2023205639,2086057648,1897238633,1976864222,1804852699,1867694188,1645340341,1724971778,1587496639,1516133128,1461550545,1406951526,1302016099,1230646740,1142491917,1087903418,2896545431,2825181984,2770861561,2716262478,3215044683,3143675388,3055782693,3001194130,2326604591,2389456536,2200899649,2280525302,2578013683,2640855108,2418763421,2498394922,3769900519,3832873040,3912640137,3992402750,4088425275,4151408268,4197601365,4277358050,3334271071,3263032808,3476998961,3422541446,3585640067,3514407732,3694837229,3640369242,1762451694,1842216281,1619975040,1682949687,2047383090,2127137669,1938468188,2001449195,1325665622,1271206113,1183200824,1111960463,1543535498,1489069629,1434599652,1363369299,622672798,568075817,748617968,677256519,907627842,853037301,1067152940,995781531,51762726,131386257,177728840,240578815,269590778,349224269,429104020,491947555,4046411278,4126034873,4172115296,4234965207,3794477266,3874110821,3953728444,4016571915,3609705398,3555108353,3735388376,3664026991,3290680682,3236090077,3449943556,3378572211,3174993278,3120533705,3032266256,2961025959,2923101090,2868635157,2813903052,2742672763,2604032198,2683796849,2461293480,2524268063,2284983834,2364738477,2175806836,2238787779,1569362073,1498123566,1409854455,1355396672,1317987909,1246755826,1192025387,1137557660,2072149281,2135122070,1912620623,1992383480,1753615357,1816598090,1627664531,1707420964,295390185,358241886,404320391,483945776,43990325,106832002,186451547,266083308,932423249,861060070,1041341759,986742920,613929101,542559546,756411363,701822548,3316196985,3244833742,3425377559,3370778784,3601682597,3530312978,3744426955,3689838204,3819031489,3881883254,3928223919,4007849240,4037393693,4100235434,4180117107,4259748804,2310601993,2373574846,2151335527,2231098320,2596047829,2659030626,2470359227,2550115596,2947551409,2876312838,2788305887,2733848168,3165939309,3094707162,3040238851,2985771188};

        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0x00000000FFFFFFFF, 0x00000000FFFFFFFF, false, 32);

        napi_value buffer;
        char* data2;
        napi_create_buffer(env, 4, (void**)(&data2), &buffer);
        int64_t i = 0;
        for(; i < 4; ++i) {
                data2[i] = sum & 0xff;
                sum >>= 8;
        }
        return buffer;
}

napi_value crc32c(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,4067132163,3778769143,324072436,3348797215,904991772,648144872,3570033899,2329499855,2024987596,1809983544,2575936315,1296289744,3207089363,2893594407,1578318884,274646895,3795141740,4049975192,51262619,3619967088,632279923,922689671,3298075524,2592579488,1760304291,2075979607,2312596564,1562183871,2943781820,3156637768,1313733451,549293790,3537243613,3246849577,871202090,3878099393,357341890,102525238,4101499445,2858735121,1477399826,1264559846,3107202533,1845379342,2677391885,2361733625,2125378298,820201905,3263744690,3520608582,598981189,4151959214,85089709,373468761,3827903834,3124367742,1213305469,1526817161,2842354314,2107672161,2412447074,2627466902,1861252501,1098587580,3004210879,2688576843,1378610760,2262928035,1955203488,1742404180,2511436119,3416409459,969524848,714683780,3639785095,205050476,4266873199,3976438427,526918040,1361435347,2739821008,2954799652,1114974503,2529119692,1691668175,2005155131,2247081528,3690758684,697762079,986182379,3366744552,476452099,3993867776,4250756596,255256311,1640403810,2477592673,2164122517,1922457750,2791048317,1412925310,1197962378,3037525897,3944729517,427051182,170179418,4165941337,746937522,3740196785,3451792453,1070968646,1905808397,2213795598,2426610938,1657317369,3053634322,1147748369,1463399397,2773627110,4215344322,153784257,444234805,3893493558,1021025245,3467647198,3722505002,797665321,2197175160,1889384571,1674398607,2443626636,1164749927,3070701412,2757221520,1446797203,137323447,4198817972,3910406976,461344835,3484808360,1037989803,781091935,3705997148,2460548119,1623424788,1939049696,2180517859,1429367560,2807687179,3020495871,1180866812,410100952,3927582683,4182430767,186734380,3756733383,763408580,1053836080,3434856499,2722870694,1344288421,1131464017,2971354706,1708204729,2545590714,2229949006,1988219213,680717673,3673779818,3383336350,1002577565,4010310262,493091189,238226049,4233660802,2987750089,1082061258,1395524158,2705686845,1972364758,2279892693,2494862625,1725896226,952904198,3399985413,3656866545,731699698,4283874585,222117402,510512622,3959836397,3280807620,837199303,582374963,3504198960,68661723,4135334616,3844915500,390545967,1230274059,3141532936,2825850620,1510247935,2395924756,2091215383,1878366691,2644384480,3553878443,565732008,854102364,3229815391,340358836,3861050807,4117890627,119113024,1493875044,2875275879,3090270611,1247431312,2660249211,1828433272,2141937292,2378227087,3811616794,291187481,34330861,4032846830,615137029,3603020806,3314634738,939183345,1776939221,2609017814,2295496738,2058945313,2926798794,1545135305,1330124605,3173225534,4084100981,17165430,307568514,3762199681,888469610,3332340585,3587147933,665062302,2042050490,2346497209,2559330125,1793573966,3190661285,1279665062,1595330642,2910671697};
        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0x00000000FFFFFFFF, 0x00000000FFFFFFFF, true, 32);
        return toBuffer(env, sum, 32);
}

napi_value crc16ibm(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,49345,49537,320,49921,960,640,49729,50689,1728,1920,51009,1280,50625,50305,1088,52225,3264,3456,52545,3840,53185,52865,3648,2560,51905,52097,2880,51457,2496,2176,51265,55297,6336,6528,55617,6912,56257,55937,6720,7680,57025,57217,8000,56577,7616,7296,56385,5120,54465,54657,5440,55041,6080,5760,54849,53761,4800,4992,54081,4352,53697,53377,4160,61441,12480,12672,61761,13056,62401,62081,12864,13824,63169,63361,14144,62721,13760,13440,62529,15360,64705,64897,15680,65281,16320,16000,65089,64001,15040,15232,64321,14592,63937,63617,14400,10240,59585,59777,10560,60161,11200,10880,59969,60929,11968,12160,61249,11520,60865,60545,11328,58369,9408,9600,58689,9984,59329,59009,9792,8704,58049,58241,9024,57601,8640,8320,57409,40961,24768,24960,41281,25344,41921,41601,25152,26112,42689,42881,26432,42241,26048,25728,42049,27648,44225,44417,27968,44801,28608,28288,44609,43521,27328,27520,43841,26880,43457,43137,26688,30720,47297,47489,31040,47873,31680,31360,47681,48641,32448,32640,48961,32000,48577,48257,31808,46081,29888,30080,46401,30464,47041,46721,30272,29184,45761,45953,29504,45313,29120,28800,45121,20480,37057,37249,20800,37633,21440,21120,37441,38401,22208,22400,38721,21760,38337,38017,21568,39937,23744,23936,40257,24320,40897,40577,24128,23040,39617,39809,23360,39169,22976,22656,38977,34817,18624,18816,35137,19200,35777,35457,19008,19968,36545,36737,20288,36097,19904,19584,35905,17408,33985,34177,17728,34561,18368,18048,34369,33281,17088,17280,33601,16640,33217,32897,16448};
        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0x0000000000000000, 0x0000000000000000, true, 16);
        return toBuffer(env, sum, 16);
}

napi_value crc16ccitt(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,4129,8258,12387,16516,20645,24774,28903,33032,37161,41290,45419,49548,53677,57806,61935,4657,528,12915,8786,21173,17044,29431,25302,37689,33560,45947,41818,54205,50076,62463,58334,9314,13379,1056,5121,25830,29895,17572,21637,42346,46411,34088,38153,58862,62927,50604,54669,13907,9842,5649,1584,30423,26358,22165,18100,46939,42874,38681,34616,63455,59390,55197,51132,18628,22757,26758,30887,2112,6241,10242,14371,51660,55789,59790,63919,35144,39273,43274,47403,23285,19156,31415,27286,6769,2640,14899,10770,56317,52188,64447,60318,39801,35672,47931,43802,27814,31879,19684,23749,11298,15363,3168,7233,60846,64911,52716,56781,44330,48395,36200,40265,32407,28342,24277,20212,15891,11826,7761,3696,65439,61374,57309,53244,48923,44858,40793,36728,37256,33193,45514,41451,53516,49453,61774,57711,4224,161,12482,8419,20484,16421,28742,24679,33721,37784,41979,46042,49981,54044,58239,62302,689,4752,8947,13010,16949,21012,25207,29270,46570,42443,38312,34185,62830,58703,54572,50445,13538,9411,5280,1153,29798,25671,21540,17413,42971,47098,34713,38840,59231,63358,50973,55100,9939,14066,1681,5808,26199,30326,17941,22068,55628,51565,63758,59695,39368,35305,47498,43435,22596,18533,30726,26663,6336,2273,14466,10403,52093,56156,60223,64286,35833,39896,43963,48026,19061,23124,27191,31254,2801,6864,10931,14994,64814,60687,56684,52557,48554,44427,40424,36297,31782,27655,23652,19525,15522,11395,7392,3265,61215,65342,53085,57212,44955,49082,36825,40952,28183,32310,20053,24180,11923,16050,3793,7920};
        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0x000000000000FFFF, 0x0000000000000000, false, 16);
        return toBuffer(env, sum, 16);
}

napi_value crc8atm(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,7,14,9,28,27,18,21,56,63,54,49,36,35,42,45,112,119,126,121,108,107,98,101,72,79,70,65,84,83,90,93,224,231,238,233,252,251,242,245,216,223,214,209,196,195,202,205,144,151,158,153,140,139,130,133,168,175,166,161,180,179,186,189,199,192,201,206,219,220,213,210,255,248,241,246,227,228,237,234,183,176,185,190,171,172,165,162,143,136,129,134,147,148,157,154,39,32,41,46,59,60,53,50,31,24,17,22,3,4,13,10,87,80,89,94,75,76,69,66,111,104,97,102,115,116,125,122,137,142,135,128,149,146,155,156,177,182,191,184,173,170,163,164,249,254,247,240,229,226,235,236,193,198,207,200,221,218,211,212,105,110,103,96,117,114,123,124,81,86,95,88,77,74,67,68,25,30,23,16,5,2,11,12,33,38,47,40,61,58,51,52,78,73,64,71,82,85,92,91,118,113,120,127,106,109,100,99,62,57,48,55,34,37,44,43,6,1,8,15,26,29,20,19,174,169,160,167,178,181,188,187,150,145,152,159,138,141,132,131,222,217,208,215,194,197,204,203,230,225,232,239,250,253,244,243};
        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0x0000000000000000, 0x0000000000000000, false, 8);
        return toBuffer(env, sum, 8);
}

napi_value crc8cdma(napi_env env, napi_callback_info info){
        size_t argsLength = 1;
        napi_value args[1];
        napi_get_cb_info(env, info, &argsLength, args, 0, 0);

        if(argsLength < 1) {
                return createFalse(env);
        }
        char* data;
        size_t dataLength;

        napi_get_buffer_info(env, args[0], (void**)(&data), &dataLength);

        uint64_t lookupTable[] = {0,155,173,54,193,90,108,247,25,130,180,47,216,67,117,238,50,169,159,4,243,104,94,197,43,176,134,29,234,113,71,220,100,255,201,82,165,62,8,147,125,230,208,75,188,39,17,138,86,205,251,96,151,12,58,161,79,212,226,121,142,21,35,184,200,83,101,254,9,146,164,63,209,74,124,231,16,139,189,38,250,97,87,204,59,160,150,13,227,120,78,213,34,185,143,20,172,55,1,154,109,246,192,91,181,46,24,131,116,239,217,66,158,5,51,168,95,196,242,105,135,28,42,177,70,221,235,112,11,144,166,61,202,81,103,252,18,137,191,36,211,72,126,229,57,162,148,15,248,99,85,206,32,187,141,22,225,122,76,215,111,244,194,89,174,53,3,152,118,237,219,64,183,44,26,129,93,198,240,107,156,7,49,170,68,223,233,114,133,30,40,179,195,88,110,245,2,153,175,52,218,65,119,236,27,128,182,45,241,106,92,199,48,171,157,6,232,115,69,222,41,178,132,31,167,60,10,145,102,253,203,80,190,37,19,136,127,228,210,73,149,14,56,163,84,207,249,98,140,23,33,186,77,214,224,123};
        uint64_t sum = crcInternalWithLookUpTable(lookupTable, data, dataLength, 0x00000000000000FF, 0x0000000000000000, false, 8);
        return toBuffer(env, sum, 8);
}

napi_value Init (napi_env env, napi_value exports) {
        napi_property_descriptor allDesc[] = {
                {"crc", 0, crc, 0, 0, 0, napi_default, 0},
                {"crc64iso", 0, crc64iso, 0, 0, 0, napi_default, 0},
                {"crc64ecma", 0, crc64ecma, 0, 0, 0, napi_default, 0},
                {"crc64", 0, crc64ecma, 0, 0, 0, napi_default, 0},
                {"crc32ieee", 0, crc32ieee, 0, 0, 0, napi_default, 0},
                {"crc32b", 0, crc32ieee, 0, 0, 0, napi_default, 0},
                {"crc32", 0, crc32ieee, 0, 0, 0, napi_default, 0},
                {"crc32c", 0, crc32c, 0, 0, 0, napi_default, 0},
                {"crc32mhash", 0, crc32mhash, 0, 0, 0, napi_default, 0},
                {"crc16ibm", 0, crc16ibm, 0, 0, 0, napi_default, 0},
                {"crc16", 0, crc16ibm, 0, 0, 0, napi_default, 0},
                {"crc16ccitt", 0, crc16ccitt, 0, 0, 0, napi_default, 0},
                {"crcccitt", 0, crc16ccitt, 0, 0, 0, napi_default, 0},
                {"crc8atm", 0, crc8atm, 0, 0, 0, napi_default, 0},
                {"crc8", 0, crc8atm, 0, 0, 0, napi_default, 0},
                {"crc8cdma", 0, crc8cdma, 0, 0, 0, napi_default, 0}
        };
        napi_define_properties(env, exports, 16, allDesc);
        return exports;
}

NAPI_MODULE(crc, Init);
